package breadboards;

import java.awt.Color;
import java.awt.Graphics;

/**
 * Subclass of GObject used for displaying line segments 
 * @author paul oser
 */
public class GLine extends GObject {
  
  public final double LINE_TOLERANCE = 1;
  
  double dx;
  double dy;
  
  Color color;
  
  //////////////////
  // constructors //
  //////////////////
  
  /**
   * constructs a zero-length segment from (0,0) to itself
   */
  public GLine() {
    this.x = 0;
    this.y = 0;
    this.dx = 0;
    this.dy = 0;
  }
  
  /**
   * constructs a segment from (x0,y0) to (x1,y1)
   * @param x0 x-coordinate of starting point
   * @param y0 y-coordinate of starting point
   * @param x1 x-coordinate of end point
   * @param y1 y-coordinate of end point
   */
  public GLine(double x0, double y0, double x1, double y1) {
    this.x = x0;
    this.y = y0;
    this.dx = x1-x;
    this.dy = y1-y;
  }
  
  /////////////
  // getters //
  /////////////
  
  /**
   * returns the bounding rectangle of the GLine
   * @return the bounding rectangle of the GLine
   */
  public GRectangle getBounds() {
    double xmin = Math.min(x, x+dx);
    double ymin = Math.min(y, y+dy);
    double width = Math.max(x, x+dx) - xmin;
    double height = Math.max(y, y+dy) - ymin;
    return new GRectangle(x,y,width,height);
  }
  
  /**
   * returns end point of the GLine
   * @return end point of the GLine
   */
  public GPoint getEndPoint() {
    return new GPoint(x+dx,y+dy);
  }
  
  /**
   * returns start point of the GLine
   * @return start point of the GLine
   */
  public GPoint getStartPoint() {
    return new GPoint(x,y);
  }
  
  /**
   * returns color of GLine
   * @return color of GLine
   */
  public Color getColor() {
    return this.color;
  }
  
  /////////////
  // setters //
  /////////////
  
  /** 
   * sets the start point for the GLine
   * @param startX x-coordinate for starting point
   * @param startY y-coordinate for starting point
   */
  public void setStartPoint(double startX, double startY) {
    this.x = startX;
    this.y = startY;
    updateBreadboard();
  }
  
  /** 
   * sets the end point for the GLine
   * @param endX x-coordinate for end point
   * @param endY y-coordinate for end point
   */
  public void setEndPoint(double endX, double endY) {
    this.dx = endX-x;
    this.dy = endY-y;
    updateBreadboard();
  }
  
  /**
   * sets the color for the GLine to the specified color
   * @param color the specified color for the GLine
   */
  public void setColor(Color color) {
    this.color = color;
    updateBreadboard();
  }
  
  ///////////////////
  // other methods //
  ///////////////////
  
  /**
   * keeping the same starting point, this scales the horizontal component of the GLine's length by sx, and the vertical component by sy
   * @param sx the horizontal scaling factor
   * @param sy the vertical scaling factor
   */
  public void scale(double sx, double sy) {
    this.dx = sx*dx;
    this.dy = sy*dy;
    updateBreadboard();
  }
  
  /**
   * keeping the same starting point, this scales the length of the GLine by sf
   * @param sf the scaling factor
   */
  public void scale(double sf) {
    scale(sf,sf);
  }
  
  /**
   * returns true if (x,y) is sufficiently close (i.e., within LINE_TOLERANCE) to the segment
   * @return true if and only if (x,y) is within LINE_TOLERANCE of the segment
   */
  public boolean contains(double px, double py) {
    if (((px > x) && (px < x+dx)) ||
        ((px > x+dx) && (px < x))) {
      if (Math.abs(dx) < 0.001) {
        return ((px - x < LINE_TOLERANCE) && 
                (((py > y) && (py < y+dy)) ||
                 ((py > y+dy) && (py < y))));
      }
      else {
        double m = dy/dx;
        double b = y - m*x;
        double yOnLine = m*x + b;
        return (Math.abs(yOnLine - y) < LINE_TOLERANCE);
      }
    }
    else {
      return false;
    }
  }

  /** 
   * draws the GLine (generally not called directly)
   */
  @Override
  public void draw(Graphics g) {
    Color originalColor = g.getColor();
    
    g.setColor(color);
    g.drawLine((int) x, (int) y, (int) (x+dx), (int) (y+dy)); 
    
    g.setColor(originalColor);
  }
  

}
