package breadboards;

import java.awt.Graphics;
import java.awt.image.BufferedImage;
import java.io.File;
import java.io.IOException;
import java.io.InputStream;

import javax.imageio.ImageIO;

import breadboard.tests.Plane;

/**
 * Subclass of GObject used for displaying a image
 * @author paul oser
 */
public class GImage extends GObject {
  
  BufferedImage image;
  
  /**
   * returns a GImage from a resource file (i.e., a .jpg or .png image file in a java package)
   * @param resourceFileName the name of the image file in your package
   * @param cls the associated class
   * @return a GImage that looks like the image file in question
   */
  public static GImage getFromResource(String resourceFileName, Class cls) {
    try {    
      GImage gImg = new GImage(new int[1][1]);
      InputStream inputStream = cls.getClassLoader().getResourceAsStream(resourceFileName);
      BufferedImage img = ImageIO.read(inputStream);
      gImg.setImage(img);
      inputStream.close();
      return gImg;
    }
    catch (IOException e) {
      return null;
    }
  }
  
  /**
   * constructs a GImage using the specified image
   * @param image the specified image
   */
  public GImage(BufferedImage image) {
    this.image = image;    
  }
  
  /**
   * constructs a GImage using the specified image at the specified location (x,y)
   * @param image the specified image
   * @param x the x-coordinate of the location specified
   * @param y the y-coordiante of the location specified
   */
  public GImage(BufferedImage image, double x, double y) {
    this.image = image;
    this.setLocation(x, y);
  }
  
  /**
   * constructs a GImage using the specified 2d array of (integer) pixel information
   * @param pixels the specified 2d array of (integer) pixel information
   */
  public GImage(int[][] pixels) {
    BufferedImage bufferedImage = new BufferedImage(pixels.length,pixels[0].length,BufferedImage.TYPE_INT_ARGB);
    for (int r = 0; r < pixels[0].length; r++) {
      for (int c = 0; c < pixels.length; c++) {
        bufferedImage.setRGB(c,r,pixels[c][r]);
      }
    }
    this.image = bufferedImage;
    //updateBreadboard();
  }
  
  /**
   * constructs a GImage based on the image identified by the given filename (and path)
   * @param filename the file containing the image to be used
   */
  public GImage(String filename) {
    setImage(filename);   
  }
  
  /**
   * reset the image of the GImage to the one identified by the given filename (and path)
   * @param filename the file containing the image to be used
   */
  public void setImage(String filename) {
    File f = null;
    
    try {
      f = new File(filename);
      image = ImageIO.read(f);
    }
    catch(IOException e) {
      System.out.println(e);
    }  
  }
  
  /**
   * reset the image of the GImage to a given (BufferedImage) image
   * @param image the image to be used as the new GImage image
   */
  public void setImage(BufferedImage image) {
    this.image = image;
  }
  
  /** 
   * saves the image associated with this GImage using the filename (and path) given
   * @param filename the name of the file (and path) to be used when saving the image
   */
  public void saveImage(String filename) {
    try {
      File f = new File(filename);
      ImageIO.write(this.image, "png", f);
    }
    catch(IOException e) {
      System.out.println(e);
    }
  }
  
  /**
   * saves the image associated with this GImage using the specified file object
   * @param file the specified file object
   */
  public void saveImage(File file) {
    try {
      ImageIO.write(this.image, "png", file);
    }
    catch(IOException e) {
      System.out.println(e);
    }
  }
  
  /** 
   * encapsulates the red, green, and blue components of a color into a single int value
   * @param red the amount of red in the color (on a scale from 0 to 255)
   * @param green the amount of green in the color (on a scale from 0 to 255)
   * @param blue the amount of blue in the color (on a scale from 0 to 255)
   * @return a single int value encapsulating the red, green, and blue components of a color
   */
  public static int createRGBPixel(int red, int green, int blue) {
    return (255<<24) | (red<<16) | (green<<8) | (blue);
  }
  
  /** 
   * encapsulates the red, green, and blue components of a color and a level of transparency into a single integer value
   * @param red the amount of red in the color (on a scale from 0 to 255)
   * @param green the amount of green in the color (on a scale from 0 to 255)
   * @param blue the amount of blue in the color (on a scale from 0 to 255)
   * @param alpha the level of transparency (on a scale from 0 to 255)
   * @return a single int value encapsulating the red, green, and blue components of a color and a level of transparency
   */
  public static int createRGBPixel(int red, int green, int blue, int alpha) {
    return (alpha<<24) | (red<<16) | (green<<8) | (blue);
  }
  
  /**
   * returns the level of transparency associated with an integer value representing a color with transparency
   * @param pixel - the integer value representing a color with transparency
   * @return - the level of transparency (0-255)
   */
  public static int getAlpha(int pixel) {
    return (pixel>>24) & 0xff;   // (shift right 24 bits, and bit-wise AND with 0xff = 255)
  }
  
  /**
   * returns the red component associated with the specified integer value
   * @param pixel - the integer value representing a color with transparency
   * @return the red component of the color (0-255)
   */
  public static int getRed(int pixel) {
    return (pixel>>16) & 0xff;   // (shift right 16 bits, and bit-wise AND with 0xff = 255)
  }
  
  /**
   * returns the green component associated with the specified integer value
   * @param pixel - the integer value representing a color with transparency
   * @return the green component of the color (0-255)
   */
  public static int getGreen(int pixel) {
    return (pixel>>8) & 0xff;    // (shift right 8 bits, and bit-wise AND with 0xff = 255)
  }
  
  /**
   * returns the blue component associated with the specified integer value
   * @param pixel - the integer value representing a color with transparency
   * @return the blue component of the color (0-255)
   */
  public static int getBlue(int pixel) {
    return pixel & 0xff;         // (no shift, just bit-wise AND with 0xff = 255)
  }
  
  /**
   * returns a 2d array of integer values corresponding to the colors of the pixels of this image
   * @return a 2d array of integer values corresponding to the colors of the pixels of this image
   */
  public int[][] getPixelArray() {
    int[][] pixels = new int[this.image.getWidth()][this.image.getHeight()];
    for (int r = 0; r < this.image.getHeight(); r++) {
      for (int c = 0; c < this.image.getWidth(); c++) {
        pixels[c][r] = this.image.getRGB(c, r);
      }
    }
    return pixels;
  }
  
  /**
   * returns the image associated with this GImage
   * @return the image associated with this GImage
   */
  public BufferedImage getImage() {
    return image;
  }

  /** 
   * draws the GImage (generally not called directly)
   */
  @Override
  public void draw(Graphics g) {
    g.drawImage(image, (int) (this.x), (int) (this.y), this.getBreadboard());
  }

  /**
   * returns true when (x,y) is inside the bounding rectangle for this GImage
   */
  @Override
  public boolean contains(double x, double y) {
    return (x >= this.getX() &&
            x <= this.getX() + this.image.getWidth() &&
            y >= this.getY() &&
            y <= this.getY() + this.image.getHeight());
  }
  
  /**
   * returns height of GImage object
   * @return the height of the GImage object
   */
  public int getHeight() {
    return this.image.getHeight();
  }
  
  /**
   * returns width of GImage object
   * @return the width of the GImage object
   */
  public int getWidth() {
    return this.image.getWidth();
  }

  /** 
   * returns the bounding rectangle for this GImage
   */
  @Override
  public GRectangle getBounds() {
    GRectangle boundingRect = new GRectangle(this.getX(),this.getY(),this.getWidth(),this.getHeight());
    return boundingRect;
  }

}
