package breadboards;

import java.awt.Graphics;
import java.util.ArrayList;

/**
 * Subclass of GObject suitable for displaying a collection of other GObjects, treated as a single entity
 * @author paul oser
 */
public class GCompound extends GObject 
                       implements GObjectParent {
  
  ArrayList<GObject> gObjects;
  
  /**
   * constructs a new (empty) GCompound
   */
  public GCompound() {
    this.gObjects = new ArrayList<GObject>();
  }
  
  /**
   * returns the ith element by z-order in this GCompound
   * @param i the z-order index
   * @return the ith element by z-order in this GCompound
   */
  public GObject getElement(int i) {
    return gObjects.get(i);
  }
  
  /**
   * returns the number of elements that make up the GCompound
   * @return the number of elements that make up the GCompound
   */
  public int getElementCount() {
    return gObjects.size();
  }
  
  /**
   * returns the top-most element of the GCompound at the specified location (x,y)
   * @param x the x-coordinate of the specified location
   * @param y the y-coordinate of the specified location
   * @return the top-most element of the GCompound at the specified location (x,y)
   */
  public GObject getElementAt(double x, double y) {
    GObject objectSought = null;
    for (GObject gObject : this.gObjects) {
      if (gObject.contains(x,y)) {
        objectSought = gObject;
      }
    }
    return objectSought;
  }
  
  /**
   * returns the top-most element of the GCompound at the specified point
   * @param pt the specified point
   * @return the top-most element of the GCompound at the specified point
   */
  public GObject getElementAt(GPoint pt) {
    return getElementAt(pt.getX(), pt.getY());
  }
  
  /**
   * converts local coordinates associated with the GCompound to coordinates associated with the parent object
   * @param x the local x-coordinate
   * @param y the local y-coordinate
   * @return coordinates in GCompound's parent object
   */
  public GPoint getCanvasPoint(double x, double y) {
    return new GPoint(x + this.x, y + this.y);
  }
  
  /**
   * converts coordinates associated with the parent object to local coordinates associated with the GCompound
   * @param x the x-coordinate associated with the parent object
   * @param y the y-coordinate associated with the parent object
   * @return the local coordinates 
   */
  public GPoint getLocalPoint(double x, double y) {
    return new GPoint(x - this.x, y - this.y);
  }
  
  /**
   * adds a GObject to this GCompound 
   * @param gObject the GObject to be added
   */
  public void add(GObject gObject) {
    this.gObjects.add(gObject); // adds at (x,y) = (0,0)
    gObject.setParent(this);
    updateBreadboard();
  }
  
  /** 
   * adds a GObject to this GCompound at the specified location
   * @param gObject the GObject to be added
   * @param x the x-coordinate of the specified location
   * @param y the y-coordiante of the specified location
   */
  public void add(GObject gObject, double x, double y) {
    gObject.setLocation(x, y);
    gObject.setParent(this);
    this.gObjects.add(gObject);
  }
  
  /**
   * adds a GObject to this GCompound at the specified point
   * @param gObject the GObject to be added
   * @param pt the specified point
   */
  public void add(GObject gObject, GPoint pt) {
    gObject.setLocation(pt.getX(),pt.getY());
    gObject.setParent(this);
    this.gObjects.add(gObject);
  }
  
  /**
   * removes the specified GObject from the GCompound
   * @param gObject the specified GObject to be removed
   */
  public void remove(GObject gObject) {
    this.gObjects.remove(gObject);
    gObject.setParent(null);
    updateBreadboard();
  }
  
  /**
   * removes all GObjects from this GCompound
   */
  public void removeAll() {
    for (int i = this.getElementCount()-1; i >= 0; i--) {
      this.gObjects.remove(i);
    }
    updateBreadboard();
  }

  /** 
   * draws the GCompound (generally not called directly)
   */
  @Override
  public void draw(Graphics g) {
    for(GObject gObject : this.gObjects) {
      if (gObject.isVisible()) {
        gObject.move(this.getX(), this.getY());
        gObject.draw(g);
        gObject.move(-this.getX(), -this.getY());
      }
    }
  }

  /**
   * returns true if the point (x,y) is contained in one of the elements of the GCompound
   * @param x the x-coordinate of the point in question
   * @param y the y-coordinate of the point in question
   */
  @Override
  public boolean contains(double x, double y) {
    boolean foundGObjectContainingPt = false;
    for(GObject gObject : this.gObjects) {
      if (gObject.contains(x, y)) 
        foundGObjectContainingPt = true;
    }
    return foundGObjectContainingPt;
  }

  /**
   * returns the bounding box for the union of all elements in this GCompound
   */
  @Override
  public GRectangle getBounds() {
    GRectangle boundingRect = new GRectangle();
    for(GObject gObject : this.gObjects) {
      if (  boundingRect.isEmpty() && 
          ! gObject.getBounds().isEmpty()) {
        boundingRect = gObject.getBounds();
      }
      else if (! boundingRect.isEmpty() && 
               ! gObject.getBounds().isEmpty())
        boundingRect = boundingRect.union(gObject.getBounds());
    }
    boundingRect.translate(this.getX(), this.getY());
    return boundingRect;
  }
  
  /**
   Swaps the z-orders of the specified GObject and the object immediately "above" it.
   * @param gObj the specified GObject
   */
  public void sendForward(GObject gObj) {
    int gObjIndex = gObjects.indexOf(gObj);
    if (gObjIndex < gObjects.size() - 1) {
      GObject nextHigherObject = gObjects.get(gObjIndex+1);
      gObjects.set(gObjIndex, nextHigherObject);
      gObjects.set(gObjIndex+1, gObj);
      updateDisplay();
    }
  }
  
  /**
   Swaps the z-orders of the specified GObject and the object immediately "below" it.
   * @param gObj the specified GObject
   */
  public void sendBackward(GObject gObj) {
    int gObjIndex = gObjects.indexOf(gObj);
    if (gObjIndex > 0) {
      GObject nextLowerObject = gObjects.get(gObjIndex-1);
      gObjects.set(gObjIndex, nextLowerObject);
      gObjects.set(gObjIndex-1, gObj);
      updateDisplay();
    }
  }
  
  /**
   * Moves the specified object "on top" of all other GObjects on the breadboard
   * @param gObj the specified GObject
   */
  public void sendToFront(GObject gObj) {
    gObjects.remove(gObj);
    gObjects.add(gObj);
    updateDisplay();
  }
  
  /**
   * Moves the specified object "below" all other GObjects on the breadboard
   * @param gObj the specified GObject
   */
  public void sendToBack(GObject gObj) {
    gObjects.remove(gObj);
    gObjects.add(0, gObj);
    updateDisplay();
  }

  /** 
   * updates the display of this GCompound and all of its parents on the breadboard drawing area
   */
  public void updateDisplay() {
    if (this.parent != null) {
      this.parent.updateDisplay();
    }
  }
  
  /**
   * returns height of GCompound
   * @return the height of GCompound
   */
  public double getHeight() {
    return this.getBounds().getHeight();
  }
  
  public double getWidth() {
    return this.getBounds().getWidth();
  }
}
